﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using RevisionAnalyser.Global;
using System.Threading;
using System.IO;

namespace RevisionAnalyser
{
    public partial class TasksForm : Form
    {
        private bool _updateSolidSX = false;
        private string _datasetPath = String.Empty;
        private int _currentTask = 0;
        private List<Task> _tasks;
        private List<ListViewItem> _messages = new List<ListViewItem>();

        protected delegate void UpdateStatusDelegate();
        protected delegate void CloseFormDelegate();
        protected delegate void EnableButtonsDelegate();
        protected delegate void UpdateSubProgressValueDelegate(int value);
        protected delegate void EnableSubProgressBarDelegate(bool value);

        public TasksForm(List<Task> tasks, bool autoClose)
        {
            InitializeComponent();
            _tasks = tasks;
            foreach (Task task in _tasks)
            {
                task.SetTasksForm(this);
            }
            uxProgressBar.Maximum = _tasks.Count;
            uxAutoCloseField.Checked = autoClose;
        }

        private void TasksForm_Load(object sender, EventArgs e)
        {
            TasksForm_SizeChanged(sender, e);
            NextTask();
        }

        public void UpdateSolidSX(string path)
        {
            _updateSolidSX = true;
            _datasetPath = path;
        }

        public void AddLog(string message, Color color)
        {
            ListViewItem item = new ListViewItem(DateTime.Now.ToLongTimeString());
            item.BackColor = color;
            item.SubItems.Add(message);
            _messages.Add(item);
        }

        public void InsertTask(Task task, int offset)
        {
            task.SetTasksForm(this);
            _tasks.Insert(_currentTask + offset, task);
            UpdateStatus();
        }

        public void EnableSubProgressBar(bool value)
        {
            if (uxSubProgressBar.InvokeRequired || uxLogView.InvokeRequired)
            {
                BeginInvoke(new EnableSubProgressBarDelegate(EnableSubProgressBar), new object[] { value });
            }
            else
            {
                if (uxSubProgressBar.Visible != value)
                {
                    uxSubProgressBar.Visible = value;
                    uxLogView.Location = value ? new Point(uxLogView.Location.X, uxLogView.Location.Y+29) : new Point(uxLogView.Location.X, uxLogView.Location.Y-29);
                    uxLogView.Size = value ? new Size(uxLogView.Size.Width, uxLogView.Size.Height-29) : new Size(uxLogView.Size.Width, uxLogView.Size.Height+29);
                }
            }
        }

        public void UpdateSubProgressValue(int value)
        {
            if (uxSubProgressBar.InvokeRequired)
            {
                BeginInvoke(new UpdateSubProgressValueDelegate(UpdateSubProgressValue), new object[] { value });
            }
            else
            {
                uxSubProgressBar.Value = value;
            }
        }

        private void UpdateStatus()
        {
            if (uxProgressBar.InvokeRequired || uxTaskLabel.InvokeRequired)
            {
                BeginInvoke(new UpdateStatusDelegate(UpdateStatus), new object[] { });
            }
            else
            {
                uxProgressBar.Maximum = _tasks.Count;
                uxProgressBar.Value = _currentTask - 1;
                uxTaskLabel.Text = (_currentTask > _tasks.Count) ? "Finished" : String.Format("Working on task {0}/{1}:", _currentTask, _tasks.Count);
                uxSaveHtmlButton.Enabled = (_currentTask > _tasks.Count);
            }
        }

        private void EnableButtons()
        {
            if (uxCloseButton.InvokeRequired || uxSaveHtmlButton.InvokeRequired)
            {
                BeginInvoke(new EnableButtonsDelegate(EnableButtons), new object[] { });
            }
            else
            {
                uxCloseButton.Enabled = true;
                uxSaveHtmlButton.Enabled = true;
            }
        }

        private void CloseForm()
        {
            if (InvokeRequired)
            {
                BeginInvoke(new CloseFormDelegate(CloseForm), new object[] { });
            }
            else
            {
                Close();
                if (_updateSolidSX)
                {
                    CurrentProject.Instance.Project.MainForm.UpdateSolidSX(_datasetPath);
                    _updateSolidSX = false;
                }
            }
        }

        public void NextTask()
        {
            _currentTask += 1;
            UpdateStatus();
            if (_currentTask <= _tasks.Count)
            {
                Task task = _tasks[_currentTask - 1];
                AddLog(String.Format("Starting {0} ({1}/{2})...", task.GetType().Name, _currentTask, _tasks.Count), Constants.COLOR_DARK_GREEN);
                Thread thread = new Thread(new ThreadStart(task.Run));
                thread.Start();
            }
            else
            {
                AddLog("Tasks finished", Constants.COLOR_DARK_GREEN);
                if (uxAutoCloseField.Checked)
                {
                    CloseForm();
                }
                else
                {
                    EnableButtons();
                }
            }
        }

        private void ExportLog()
        {
            if (uxSaveHtmlDialog.ShowDialog() == DialogResult.OK)
            {
                TextWriter tw = new StreamWriter(uxSaveHtmlDialog.FileName);
                try
                {
                    tw.WriteLine("<html><head><title>Dependancy Analyser Log File</title><style>");
                    tw.WriteLine("body, table, td { font-family: Verdana, sans; font-size: 11px; color: #333; border: 0; }");
                    tw.WriteLine("h1 { font-size: 1.6em; margin-bottom: 0.5em; }");
                    tw.WriteLine("th { background: #555; color: #fff; font-weight: bold; text-align: left; padding: 0.3em; }");
                    tw.WriteLine("td { border-top: 0.2em solid #fff; padding: 0.3em; }");
                    tw.WriteLine("</style></head><body><h1>Dependancy Analyser Log (" + DateTime.Now.ToLongDateString() + ")</h1>");
                    tw.WriteLine("<table width='100%' cellpadding='0' cellspacing='0'><tr><th width='10%'>Time</th><th>Message</th></tr>");
                    foreach (ListViewItem item in uxLogView.Items)
                    {
                        tw.WriteLine("<tr style='background-color: " + ColorTranslator.ToHtml(item.BackColor) + "'>");
                        tw.WriteLine("<td>" + item.Text + "</td><td>" + Functions.HTMLEncodeSpecialChars(item.SubItems[1].Text) + "</td></tr>");
                    }
                    tw.WriteLine("</body></html>");
                }
                finally
                {
                    tw.Close();
                    if (MessageBox.Show("Log file successfully created. Do you want to open it now?", "Log file created", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
                    {
                        System.Diagnostics.Process.Start(uxSaveHtmlDialog.FileName);
                    }
                }
            }
        }

        private void uxLogTimer_Tick(object sender, EventArgs e)
        {
            uxLogTimer.Enabled = false;
            try
            {
                if (_messages.Count > 0)
                {
                    uxLogView.BeginUpdate();
                    uxLogView.Items.AddRange(_messages.ToArray());
                    uxLogView.Items[uxLogView.Items.Count - 1].EnsureVisible();
                    uxLogView.EndUpdate();
                }
            }
            finally
            {
                _messages.Clear();
                uxLogTimer.Enabled = true;
            }
        }

        private void TasksForm_SizeChanged(object sender, EventArgs e)
        {
            uxEventHeader.Width = Width - 170;
        }

        private void uxSaveHtmlButton_Click(object sender, EventArgs e)
        {
            ExportLog();
        }

        private void uxCloseButton_Click(object sender, EventArgs e)
        {
            CloseForm();
        }
    }
}
